import { Tags } from "./interfaces.js";
/**
 * Associates tags with their respective colors.
 * @param tag The input tag whose color we want.
 * @returns A string representing a CSS class with the appropriate color.
 */
const tagToColor = (tag) => {
    switch (tag) {
        case Tags.Death:
        case Tags.Violence:
        case Tags.Sex:
        case Tags.Gore:
        case Tags.Profanity:
            return "red";
        case Tags.Dark:
        case Tags.Horror:
            return "darkRed";
        case Tags.SciFi:
        case Tags.Sad:
        case Tags.SoL:
        case Tags.Random:
        case Tags.Tragedy:
        case Tags.Anthology:
            return "blue";
        case Tags.Romance:
            return "romance";
        case Tags.Adventure:
            return "adventure";
        case Tags.Comedy:
            return "comedy";
        case Tags.Drama:
            return "drama";
        case Tags.Crossover:
        case Tags.Human:
        case Tags.AU:
        // We let this fall through, since these simply use the neutral color anyways.
        default:
            return "neutral";
    }
};
/**
 * Calculates the Wilson-score of a fanfic.
 * Math taken from: https://github.com/msn0/wilson-score-interval
 * @param fic The fanfic whose score we want.
 * @returns The Wilson-score of the fanfic.
 */
export const wilsonScore = (fic) => {
    const [positiveScore, total] = [fic.upVotes, fic.upVotes + fic.downVotes];
    // phat is the proportion of successes
    // in a Bernoulli trial process
    const phat = positiveScore / total;
    // z is 1-alpha/2 percentile of a standard
    // normal distribution for error alpha=5%
    const z = 1.96;
    // implement the algorithm
    // (http://goo.gl/kgmV3g)
    const a = phat + z * z / (2 * total);
    const b = z * Math.sqrt((phat * (1 - phat) + z * z / (4 * total)) / total);
    const c = 1 + z * z / total;
    return (a - b) / c;
};
/**
 * Converts Fimfic's formatting into HTML tags that can be rendered.
 * @param desc The input description with Fimfic formatting.
 * @returns The same description with HTML tags.
 */
const convertFormatting = (desc) => {
    const replacements = [
        [/\[url=(.*?)](.*?)\[\/url]/g, "<a href='$1'>$2</a>"],
        [/\[b](.*?)\[\/b]/g, "<b>$1</b>"],
        [/\[u](.*?)\[\/u]/g, "<u>$1</u>"],
        [/\[i](.*?)\[\/i]/g, "<i>$1</i>"],
        [/\[quote](.*?)\[\/quote]/g, "<blockquote>$1</blockquote>"],
        [/\[hr]/g, "<hr>"],
        [/\[color=(.*?)](.*?)\[\/color]/g, "<span style='color: $1'>$2</span>"],
        [/\[size=(.*?)](.*?)\[\/size]/g, "<span style='font-size: $1'>$2</span>"],
    ];
    return replacements.reduce((input, replacer) => input.replace(replacer[0], replacer[1]), desc);
};
/**
 * Converts a fanfic into a Div element.
 * @param fic The fanfic we want to convert into a Div element.
 * @returns A Div element populated with the fanfic's details.
 */
const convertToDiv = (fic) => {
    const convertTag = (tag) => `<p class="tag ${tagToColor(tag)}">${tag}</p>`;
    const div = document.createElement("div");
    div.className = "entry";
    // Apparently this is somewhat slower than creating the elements yourself, but it conveys the
    // meaning of the code far better and I personally couldn't notice any slowdown.
    div.innerHTML = `
        <a class="title" href="${fic.url}"><p>${fic.name}</p></a>

        <div class="flexer">
            ${(fic.image != "undefined") ? `<a href="${fic.url}"><img src="${fic.image}"></a>` : ""}
        </div>

        <div class="description">
            <details>
            ${convertFormatting(fic.desc)}
            <summary>Description</summary>
            </details>
        </div>

        <div class="reviews">
            <details>
            ${fic.reviews.map(p => "<p>" + p + "</p>").join("<hr>\n")}
            <summary>Reviews</summary>
            </details>
        </div>

        <div class="categories">
            ${fic.tags.map(convertTag).join("\n")}
        </div>

        <div class="footer">
            <span class="rating ${fic.rating}">${fic.rating}</span>
            <span title="Wilson-score" class="wilson">${Math.round(wilsonScore(fic) * 1000)}</span>
            <span Words" class="length">${Math.round(fic.length / 1000)}k Words</span>
            <span class="upVotes">${fic.upVotes}</span>
            <span class="downVotes">${fic.downVotes}</span>
            <span class="recs">${fic.recommends} Recs.</span>
        </div>
    `;
    return div;
};
/**
 * Takes a list of fanfics and converts them into DOM elements before rendering them onto the page.
 * @param ficList The list of fanfics we want to render.
 */
export const attachMainList = (ficList) => {
    const root = document.querySelector("#root");
    const ficDivs = ficList.map(convertToDiv);
    // Should never happen, but the typechecker is complaining.
    if (root == null) {
        throw new Error("The root element is missing!");
    }
    // We use replaceChildren since it both removes all previous elements and allows us to add new ones.
    root.replaceChildren(...ficDivs);
};
/**
 * Takes the list of secondary nominations and renders them as a <li> list.
 */
export const attachSecondaryList = (list) => {
    const singleRoot = document.querySelector("#singles ul");
    // Should never happen, but the typechecker is complaining.
    if (singleRoot == null) {
        throw new Error("The div for single nominations was missing!");
    }
    singleRoot.innerHTML = "";
    list.forEach(([title, url]) => {
        const li = document.createElement("li");
        li.innerHTML = `<a href="${url.toString()}">${title}</a>`;
        singleRoot.appendChild(li);
    });
};
