import { fanfics, singleNominations } from "./fanficList.js";
import { Tags } from "./interfaces.js";
import { attachMainList, attachSecondaryList, wilsonScore } from "./listGenerator.js";
/**
 * Converts the user's inputs into a SortOptions struct.
 * @returns {SortOptions}
 */
const getSortOptions = () => {
    const sortInput = document.querySelector("input[name=sortVersion]:checked");
    const sortDirectionInput = document.querySelector("input[name=sortDirection]:checked");
    if (sortInput === null || sortDirectionInput === null) {
        throw new Error("Sorting input or sorting direction was not found.");
    }
    const sortBy = sortInput.value;
    const sortDirection = sortDirectionInput.value;
    // Should never actually happen, but it shuts up the TS typechecker.
    if (!(sortBy === "wordcount" || sortBy === "upvotes" || sortBy === "recommendations" || sortBy === "wilson")) {
        throw new Error("Sorting input was malformed or missing: " + sortBy);
    }
    if (!(sortDirection == "ascending" || sortDirection == "descending")) {
        throw new Error("Sorting direction was malformed or missing:" + sortDirection);
    }
    // Converts checkboxes to True or False values and places them into a Map [Drama => True, Adventure => False, etc.]
    // Also does the same to tags.
    const tags = [...document.querySelectorAll("#filter #tags input, #filter #genres input")].reduce((map, elem) => { map.set(elem.value, elem.checked); return map; }, new Map());
    return {
        sortBy,
        tags,
        sortDirection
    };
};
/**
 * Filters and sorts a list of fanfics based on the provided options.
 * @param list The input list of fanfics.
 * @param options The sorting options generated by getSortOptions.
 * @returns {iFanfic[]} The filtered and sorted list of fanfics.
 */
const filterList = (list, options) => {
    // If two elements happen to have the same value we get a 0 during sorting. To break the tie, we
    // fall back to alphabetical sorting and this small function helps with this.
    const ifZeroReplaceWith = (val, replacement) => (val != 0) ? val : replacement;
    const sortFunctions = {
        "wordcount": (a, b) => ifZeroReplaceWith(b.length - a.length, a.name.localeCompare(b.name)),
        "upvotes": (a, b) => ifZeroReplaceWith(b.upVotes - a.upVotes, a.name.localeCompare(b.name)),
        "recommendations": (a, b) => ifZeroReplaceWith(b.recommends - a.recommends, a.name.localeCompare(b.name)),
        "wilson": (a, b) => ifZeroReplaceWith(wilsonScore(b) - wilsonScore(a), a.name.localeCompare(b.name)),
    };
    const newList = [...list]
        .filter(f => f.tags.some(tag => options.tags.get(tag)))
        .sort((a, b) => sortFunctions[options.sortBy](a, b));
    if (options.sortDirection == "ascending") {
        newList.reverse();
    }
    return newList;
};
const filterSecondaries = (list, options) => {
    return list.filter(f => f[2].some(t => options.tags.get(t)));
};
/**
 * Generates list elements from the enums and hooks them up with controls.
 */
export const hookControls = () => {
    const genreDiv = document.querySelector("#filter #genres div");
    const tagDiv = document.querySelector("#filter #tags div");
    // Should never happen, but the typechecker is complaining.
    if (genreDiv == null || tagDiv == null) {
        throw new Error("The genre- or tag-container div is missing!");
    }
    // ADD GENRES TO THE DIV
    const genres = [
        Tags.Romance,
        Tags.Comedy,
        Tags.Adventure,
        Tags.SoL,
        Tags.Sad,
        Tags.Tragedy,
        Tags.Random,
        Tags.Dark,
    ];
    for (const genre of genres) {
        if (typeof (genre) === "string") {
            const label = document.createElement("label");
            label.htmlFor = genre;
            label.innerText = genre;
            genreDiv.appendChild(label);
            const elem = document.createElement("input");
            elem.type = "checkbox";
            elem.value = genre;
            elem.id = genre;
            elem.name = "genre";
            elem.checked = true;
            genreDiv.appendChild(elem);
        }
    }
    // ADD TAGS TO THE DIV
    const tagElems = [];
    for (const tag of Object.values(Tags)) {
        if (typeof (tag) === "string" && genres.indexOf(tag) == -1) {
            const label = document.createElement("label");
            label.htmlFor = tag;
            label.innerText = tag;
            tagDiv.appendChild(label);
            const elem = document.createElement("input");
            elem.type = "checkbox";
            elem.value = tag;
            elem.id = tag;
            elem.name = "tag";
            elem.checked = true;
            // This is a bit of a hack, but it works reasonably well. Adds a hover tooltip to the AU tag.
            if (tag == "AU") {
                label.title = "Alternative Universe";
            }
            tagDiv.appendChild(elem);
            tagElems.push(elem);
        }
    }
    // ADD BUTTONS AFTER TAGS
    const allButton = document.createElement("button");
    allButton.innerText = "Check all";
    const noneButton = document.createElement("button");
    noneButton.innerText = "Check none";
    const buttonsDiv = document.getElementById("buttons");
    // This should never happen, since it's part of the HTML file, but the typechecker is complaining.
    if (buttonsDiv == null) {
        throw new Error("The div for the check all/none buttons is missing!");
    }
    else {
        buttonsDiv.appendChild(noneButton);
        buttonsDiv.appendChild(allButton);
    }
    // HOOK EVERYTHING TO CHANGE THE LIST WHENEVER THEY'RE CHANGED.
    const update = () => {
        attachMainList(filterList(fanfics, getSortOptions()));
        attachSecondaryList(filterSecondaries(singleNominations, getSortOptions()));
    };
    document.querySelectorAll("#filter input").forEach(elem => elem.addEventListener("change", update));
    noneButton.addEventListener("click", () => {
        tagElems.forEach(elem => { elem.checked = false; });
        update();
    });
    allButton.addEventListener("click", () => {
        tagElems.forEach(elem => { elem.checked = true; });
        update();
    });
};
